<?php

namespace App\Http\Controllers\Superadmin\Persona;

use App\Entities\Area;
use App\Entities\Cargo;
use App\Entities\Persona;
use App\Entities\Sede;
use App\Entities\User;
use App\Entities\UserPersona;
use App\Http\Requests\PersonaRequest;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;

class PersonaController extends Controller
{
    protected $userId;

    /**
     * PersonaController constructor.
     */
    public function __construct()
    {
        $this->userId = Session::get('usuario.id');
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $styles = [
            'assets/css/dataTables.bootstrap'
        ];
        $scripts = [
            'assets/js/datatable/jquery.dataTables.min',
            'assets/js/datatable/ZeroClipboard',
            'assets/js/datatable/dataTables.tableTools.min',
            'assets/js/datatable/dataTables.bootstrap.min',
            'assets/js/inputmask/jasny-bootstrap.min',
            'assets/js/select2/select2.min',
            'assets/js/select2/select2es',
            'support/js/plugins/clipboard/clipboard.min'
        ];

        $personas = Persona::with('cargo')->orderBy('id', 'DESC')->get();
        foreach ($personas as $key => $persona) {
            $personas[$key]->area = Area::with('sede')->find($persona->cargo->area_id);
        }
//        dd($personas[0]);

        return view('superadmin.personas.index', compact('styles', 'scripts', 'personas'));
    }

    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($slug)
    {
        $scripts = [
            'assets/js/inputmask/jasny-bootstrap.min',
            'assets/js/select2/select2.min',
            'assets/js/select2/select2es'];

        $estadoCivil = Config::get('constants.estadoCivil');
        $genero = Config::get('constants.genero');

        $persona = Persona::where('slug', $slug)->first();
        $cargo = Cargo::find($persona->cargo_id);
        $area = Area::find($cargo->area_id);

        $persona->sede_id = $area->sede_id;
        $persona->area_id = $cargo->area_id;
        $persona->fecha_nacimiento = formatDate($persona->fecha_nacimiento);

        $sedes = Sede::pluck('nombre', 'id')->all();
        $areas = Area::where('sede_id', $area->sede_id)->pluck('nombre', 'id')->all();
        $cargos = Cargo::where('area_id', $cargo->area_id)->pluck('nombre', 'id')->all();

        return view('superadmin.personas.show', compact('styles', 'scripts', 'estadoCivil', 'genero', 'sedes', 'areas', 'cargos', 'persona'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create()
    {
        $scripts = [
            'assets/js/inputmask/jasny-bootstrap.min',
            'assets/js/select2/select2.min',
            'assets/js/select2/select2es'];

        $estadoCivil = Config::get('constants.estadoCivil');
        $genero = Config::get('constants.genero');

        $sedes = Sede::pluck('nombre', 'id')->all();
        $areas = [];
        $cargos = [];

        return view('superadmin.personas.create', compact('styles', 'scripts', 'estadoCivil', 'genero', 'sedes', 'areas', 'cargos'));

    }

    /**
     * @param PersonaRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(PersonaRequest $request)
    {
        /**
         * Create Persona
         */
        $persona = new Persona($request->all());
        $persona->fecha_nacimiento = formatDate($request->fecha_nacimiento);
        $persona->apellidos_nombres = mb_strtoupper($request->apellidos_nombres);
        $foto = $request->file('foto');
        $pictureOld = ($request->foto == null) ? null : $request->foto;
        if (isset($foto) && $foto != null) {
            $persona->foto = loadImagePersona($foto, $pictureOld);
        }
        $persona->save();

        /**
         * Create User
         */
        $user = new User();
        $user->username = $persona->dni;
        $user->last_first_name = $persona->apellidos_nombres;
        $user->email = $persona->email;
        $user->password = $persona->dni;
        $user->role_id = 3;
        $user->remember_token = str_random(10);
        $user->save();

        /**
         * Create UserPersona
         */
        $userPersona = new UserPersona();
        $userPersona->user_id = $user->id;
        $userPersona->persona_id = $persona->id;
        $userPersona->save();

        /**
         * Send message to response
         */
        $message = 'Se creo la persona y su usuario correctamente';
        messageFlash($message, 'success');
        return redirect()->route('superadmin.personas.index');
    }

    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit($slug)
    {
        $scripts = [
            'assets/js/inputmask/jasny-bootstrap.min',
            'assets/js/select2/select2.min',
            'assets/js/select2/select2es'
        ];

        $estadoCivil = Config::get('constants.estadoCivil');
        $genero = Config::get('constants.genero');

        $persona = Persona::where('slug', $slug)->first();

        $cargo = Cargo::find($persona->cargo_id);
        $area = Area::find($cargo->area_id);

        $persona->sede_id = $area->sede_id;
        $persona->area_id = $cargo->area_id;
        $persona->fecha_nacimiento = formatDate($persona->fecha_nacimiento);

        $sedes = Sede::pluck('nombre', 'id')->all();
        $areas = Area::where('sede_id', $area->sede_id)->pluck('nombre', 'id')->all();
        $cargos = Cargo::where('area_id', $cargo->area_id)->pluck('nombre', 'id')->all();

        return view('superadmin.personas.edit', compact('styles', 'scripts', 'estadoCivil', 'genero', 'sedes', 'areas', 'cargos', 'persona'));

    }

    /**
     * @param PersonaRequest $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(PersonaRequest $request, $id)
    {
        
        $data = $request->all();
        $data['fecha_nacimiento'] = formatDate($request->fecha_nacimiento);
        $data['apellidos_nombres'] = mb_strtoupper($request->apellidos_nombres);
        $foto = $request->file('foto');
        $pictureOld = ($request->foto_old == null) ? null : $request->foto_old;
        if (isset($foto) && $foto != null) {
            $data['foto'] = loadImagePersona($foto, $pictureOld);
        }
        $persona = Persona::find($id);
        $persona->slug = null;
        $persona->update($data);

        //Actualizar usuario
        $userPersona = UserPersona::where('persona_id', $persona->id)->first();
        
        $user = User::find($userPersona->user_id);
        
        $user->username = $persona->dni;
        $user->slug = null;
        $user->email = $persona->email;
        $user->last_first_name = $persona->apellidos_nombres;
        
        $user->save();

        $message = 'Se actualizo la persona correctamente';
        messageFlash($message, 'success');

        return redirect()->route('superadmin.personas.index');
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        if ($request->ajax()) {
            $persona = Persona::find($id);
            $persona->delete();
            $response = [
                'message' => "Se elimino el personal correctamente",
                'route' => route('superadmin.personas.index')
            ];
            return response()->json(['data' => $response]);
        }
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function cumpleanios()
    {
        $dateMonth = date('m');
        $personas = Persona::select('*', DB::raw('DAY(fecha_nacimiento) as dateDay'))
            ->whereMonth('fecha_nacimiento', '=', $dateMonth)
            ->orderBy('dateDay')
            ->get();

        return view('superadmin.personas.cumpleanios.index', compact('styles', 'scripts', 'personas'));
    }

    /**
     * @param Request $request
     * @param $slug
     * @return \Illuminate\Http\RedirectResponse
     */
    public function cumpleaniosPersonal(Request $request, $slug)
    {
        $persona = Persona::with('cargo')->where('slug', $slug)->first();
        if ($persona) {

            $cargo = Cargo::find($persona->cargo_id);
            $area = Area::find($cargo->area_id);
            $persona->area = $area;
            $persona->sede = Sede::find($area->sede_id);

            return view('superadmin.personas.cumpleanios.show', compact('persona'));
        } else {
            $message = 'No se encontro resultado para este personal.';
            messageFlash($message, 'danger');
            return redirect()->route('superadmin.cumpleanios.index');
        }
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function cumpleanioEnvio(Request $request, $id)
    {
        if ($request->ajax()) {
            $persona = Persona::find($id);
            if ($persona) {
                Artisan::call("email:birthday", ['id' => $id]);

                $response = [
                    'route' => route('superadmin.cumpleanios.index'),
                    'message' => "Se envío el saludo de Cumpleaño a " . $persona->apellidos_nombres
                ];
                return response()->json(['data' => $response], 200);
            }
            $response = [
                'message' => "No se puede enviar el saludo de Bienvenida, porque no existe personal."
            ];
            return response()->json($response, 200);
        }
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function welcome(Request $request, $id)
    {
        if ($request->ajax()) {
            $persona = Persona::find($id);
            if ($persona) {
                Artisan::call("email:welcome", ['dni' => $persona->dni]);

                $response = [
                    'message' => "Se envío el saludo de Bienvenido a " . $persona->apellidos_nombres
                ];
                return response()->json(['data' => $response], 200);
            }
            $response = [
                'message' => "No se puede enviar el saludo de Bienvenida, porque no existe personal."
            ];
            return response()->json($response, 200);
        }
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function firmaCorreo(Request $request, $id)
    {
        if ($request->ajax()) {
            $persona = Persona::with('cargo')->find($id);
            return view('superadmin.personas.firmacorreo', compact('persona'));
        }
    }

    /**
     * @param Request $request
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\Http\RedirectResponse|\Illuminate\View\View
     */
    public function tarjetapermiso(Request $request, $slug)
    {
        $persona = Persona::with('cargo')->where('slug', $slug)->first();
        if ($persona) {

            $cargo = Cargo::find($persona->cargo_id);
            $area = Area::find($cargo->area_id);
            $persona->area = $area;
            $persona->sede = Sede::find($area->sede_id);

            return view('superadmin.personas.cumpleanios.tarjetapermiso', compact('persona'));
        } else {
            $message = 'No se encontro resultado para este personal.';
            messageFlash($message, 'danger');
            return redirect()->route('superadmin.cumpleanios.index');
        }
    }

}
