<?php

namespace App\Http\Controllers\Admin\Persona;

use App\Entities\Area;
use App\Entities\Cargo;
use App\Entities\Persona;
use App\Entities\PersonaMessage;
use App\Entities\Sede;
use App\Entities\UserPersona;
use App\Http\Requests\PersonaBasicaRequest;
use App\Http\Requests\PersonaContactRequest;
use Illuminate\Http\Request;

use App\Http\Requests;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;

class PersonaController extends Controller
{
    /**
     * PersonaController constructor.
     */
    public function __construct()
    {

    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
//        $styles = ['vendors/bower_components/chosen/chosen.min'];
//        $scripts = ['vendors/bower_components/bootstrap-growl/jquery.bootstrap-growl.min'];

        return view('admin.personas.index', compact('styles', 'scripts'));
    }

    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show($slug)
    {
        $persona = Persona::with('cargo')->where('slug', $slug)->first();
        if ($persona) {
            $persona->fechaNacimiento = $persona->fecha_nacimiento;
            $persona->fecha_nacimiento = dateCarbon($persona->fecha_nacimiento);
            $persona->area = Area::find($persona->cargo->area_id);
            $persona->sede = Sede::find($persona->area->sede_id);
            $persona->gener = $persona->genero;
            $persona->genero = ($persona->genero == "M") ? "Masculino" : "Femenino";

            return view('admin.personas.show', compact('slug', 'persona'));
        } else {
            $message = 'No se encontro resultado para este personal.';
            messageFlash($message, 'danger');

            return redirect()->route('admin.persona.index');
        }
    }

    /**
     * @param $slug
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function companieros($slug)
    {
        $persona = Persona::with('cargo')->where('slug', $slug)->first();
        $persona->fechaNacimiento = $persona->fecha_nacimiento;
        $persona->gener = $persona->genero;
        $cargos = Cargo::where('area_id', $persona->cargo->area_id)->get();
        $personas = [];

        foreach ($cargos as $key => $cargo) {
            $gentes = Persona::where([['cargo_id', $cargo->id], ['id', '!=', $persona->id]])->get();
            foreach ($gentes as $jey => $gente) {
                $personas[] = $gente;
            }
        }

        return view('admin.personas.companieros', compact('slug', 'cargos', 'persona', 'personas'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function perfil()
    {
        $styles = ['vendors/bower_components/eonasdan-bootstrap-datetimepicker/build/css/bootstrap-datetimepicker.min'];
        $scripts = [
            'vendors/bower_components/moment/min/moment-with-locales.min',
            'vendors/bower_components/eonasdan-bootstrap-datetimepicker/build/js/bootstrap-datetimepicker.min',
            'vendors/fileinput/fileinput.min'
        ];

        $userPersona = UserPersona::with('persona')->where('user_id', Auth::guard('admin')->user()->id)->first();
        $persona = $userPersona->persona;
        $persona->fechaNacimiento = $persona->fecha_nacimiento;
        $persona->fecha_nacimiento = formatDate($persona->fecha_nacimiento);
        $persona->area = Area::find($persona->cargo->area_id);
        $persona->sede = Sede::find($persona->area->sede_id);
        $persona->gener = $persona->genero;
        $persona->genero = ($persona->genero == "M") ? "MASCULINO" : "FEMENINO";
        $estadoCivil = Config::get('constants.estadoCivil');
        $genero = Config::get('constants.genero');

        return view('admin.personas.perfil.index', compact('styles', 'scripts', 'persona', 'genero', 'estadoCivil'));
    }

    /**
     * @param PersonaBasicaRequest $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateBasic(PersonaBasicaRequest $request, $id)
    {
        $data = $request->all();
        $data['fecha_nacimiento'] = formatDate($request->fecha_nacimiento);
        $persona = Persona::find($id);
        $persona->slug = null;
        $persona->update($data);

        $message = 'Se actualizo la información de contacto correctamente';
        messageFlash($message, 'success');
        return response()->json(
            [
                'message' => $message,
                'route' => route('admin.perfil.index')
            ], 200);
    }

    /**
     * @param PersonaContactRequest $request
     * @param $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateContact(PersonaContactRequest $request, $id)
    {
        $persona = Persona::find($id);
        $persona->update($request->all());

        $message = 'Se actualizo la información de contacto correctamente';
        messageFlash($message, 'success');
        return response()->json(
            [
                'message' => $message,
                'route' => route('admin.perfil.index')
            ], 200);
    }

    /**
     * @param Request $request
     * @param $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function foto(Request $request, $id)
    {
        $file = $request->file('foto');
        if (isset($file) && $file != null) {

            $persona = Persona::find($id);
            $pictureOld = ($persona->foto == null) ? null : $persona->foto;
            $persona->foto = $this->isFileProfile($file, $pictureOld);
            $persona->update();

            $message = 'Se subio correctamente la imagen';
            $type = "success";

        } else {

            $message = 'No se pudo subir la imagen, intentelo otra ves';
            $type = "danger";

        }
        messageFlash($message, $type);

        return redirect()->route('admin.perfil.index');
    }

    /**
     * @param $file
     * @param null $pictureOld
     * @return string
     */
    private function isFileProfile($file, $pictureOld = null)
    {
        if ($pictureOld != null) {
            if (Storage::disk('perfil')->exists($pictureOld)) {
                Storage::disk('perfil')->delete($pictureOld);
                Storage::disk('perfilThumbnail')->delete('thumb-' . $pictureOld);
            }
        }
        $slug = date('Y') . '-' . date('m') . '-' . date('d') . '-' . uniqid();
        $fileName = $slug . '.jpg';

        $thumbnail = public_path('web/perfil/thumbnail');

        $img = Image::make($file->getRealPath());
        $img->resize(250, 250);
        $img->save($thumbnail . '/thumb-' . $fileName);

        Storage::disk('perfil')->put($fileName, file_get_contents($file->getRealPath()));

        return $fileName;
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function cumpleanios()
    {
        $dateMonth = date('m');
        $personas = Persona::select('*', DB::raw('DAY(fecha_nacimiento) as dateDay'))
            ->whereMonth('fecha_nacimiento', '=', $dateMonth)
            ->orderBy('dateDay')
            ->get();

        return view('admin.personas.cumpleanios.index', compact('styles', 'scripts', 'personas'));
    }

    /**
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function cumpleaniosMessage()
    {
        $user = Auth::guard('admin')->user();
        $userPersona = UserPersona::with('persona')->where('user_id', $user->id)->first();
        $persona = $userPersona->persona;
        $persona->fechaNacimiento = $persona->fecha_nacimiento;
        $persona->gener = $persona->genero;

        $messages = PersonaMessage::with('persona')
            ->where('persona_cumpleanio_id', $persona->id)
            ->orderBy('created_at','DESC')
            ->get();

        return view('admin.personas.perfil.cumpleanios', compact('styles', 'scripts', 'persona', 'messages'));
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function cumpleaniosSendMessage(Request $request)
    {
        if ($request->ajax()) {
            $user = Auth::guard('admin')->user();
            $userPersona = UserPersona::with('persona')->where('user_id', $user->id)->first();
            $persona = $userPersona->persona;

            $personaMessage = new PersonaMessage();
            $personaMessage->message = $request->data['message'];
            $personaMessage->persona_id = $persona->id;
            $personaMessage->persona_cumpleanio_id = $request->data['persona_cumpleanio_id'];
            $personaMessage->save();

            $response = [
                'message' => "Se envío tu saludo correctamente."
            ];

            return response()->json(['data' => $response]);
        }
    }
}
